import LicenseSpring

import SwiftUI
import UniformTypeIdentifiers

struct InitialView: View {
    @State private var licenseKey = ""
    @State private var user = ""
    @State private var password = ""
    @State private var trialID = ""
    @State private var serverId = ""
    @State private var airgapLicenseKey = ""
    @State private var initializationCode = ""
    @State private var activationCode = ""
    @State private var confirmationCode = ""
    @State private var policyFile: URL?
    @State private var policyID = ""
    
    @State private var bundleLicenseKey = ""
    @State private var bundleUser = ""
    @State private var bundlePassword = ""
    @State private var bundleServerId = ""
    
    @State private var openFilePresented = false
    @State private var saveFilePresented = false
    @State private var saveFileData: Data?
    
    let manager: LicenseManager!
    let bundleManager: BundleManager!
    @Binding var showConfiguration: Bool
    
    @EnvironmentObject private var viewUpdater: ViewUpdaterHack
    
    var body: some View {
        InitialViewContainer(manager: manager, showConfiguration: $showConfiguration, showProductDetails: true) {
            if let manager = manager, manager.configuration.isVMDetectionEnabled {
                Text("Virtual machine \(manager.configuration.isVM ? "detected" : "not detected")")
                    .foregroundColor(.red)
                    .fontWeight(.bold)
                    .frame(alignment: .leading)
            }
            GroupBox(label: Text("Online Activation")) {
                HStack(alignment: .top) {
                    VStack {
                        Text("Key-based product")
                        TextField("License Key", text: $licenseKey)
                        TextField("Invisible", text: .constant(""))
                            .opacity(0)
                        MinWidthButton("Activate") {
                            performAction(name: "Activate License", successAlert: false) {
                                _ = try manager.activateLicense(licenseKey: licenseKey)
                            }
                        }
                        .disabled(licenseKey.isEmpty)
                        MinWidthButton("Create Request") {
                            guard let location = locationForFile(save: true) else { return }
                            performAction(name: "Create Offline Activation") {
                                _ = try manager.createOfflineActivationFile(licenseKey: licenseKey, preferredLocation: location)
                            }
                        }
                        .disabled(licenseKey.isEmpty)
                    }
                    Divider()
                    VStack {
                        Text("User-based product")
                        TextField("User", text: $user)
                        TextField("Password", text: $password)
                        TextField("Server ID", text: $serverId)
                        MinWidthButton("Activate") {
                            performAction(name: "Activate License", successAlert: false) {
                                if let serverId = Int64(serverId){
                                    _ = try manager.activateLicense(user: user, password: password, serverId: serverId)
                                }
                                else {
                                    _ = try manager.activateLicense(user: user, password: password)
                                }
                            }
                        }
                        .disabled(user.isEmpty || password.isEmpty)
                        MinWidthButton("Create Request") {
                            guard let location = locationForFile(save: true) else { return }
                            performAction(name: "Create Offline Activation") {
                                _ = try manager.createOfflineActivationFile(user: user, password: password, preferredLocation: location, serverID: Int64(serverId) ?? 0)
                            }
                        }
                        .disabled(user.isEmpty || password.isEmpty)
                    }
                    Divider()
                    VStack {
                        Text("Get trial license")
                        TextField("Customer ID (usually email)", text: $trialID)
                        TextField("Invisible", text: .constant(""))
                            .opacity(0)
                        MinWidthButton("Get Trial") {
                            performAction(name: "Get Trial", successAlert: false) {
                                licenseKey = try manager.requestTrialKey(with: trialID)
                            }
                        }
                        .disabled(trialID.isEmpty)
                    }
                }
            }
            GroupBox(label: Text("Offline Activation")) {
                HStack(alignment: .top) {
                    MinWidthButton("Open Activation Page") {
                        NSWorkspace.shared.open(URL(string: "https://saas.licensespring.com/offline/")!)
                    }
                    MinWidthButton("Activate Offline") {
                        guard let location = locationForFile(save: false) else { return }
                        performAction(name: "Activate Offline") {
                            _ = try manager.activateLicenseOffline(location)
                        }
                    }
                }
                .frame(maxWidth: .infinity)
            }
            GroupBox(label: Text("SSO Activation")) {
                HStack(alignment: .top) {
                    MinWidthButton("Get SSO URL") {
                        performAction(name: "Actiave with SSO", successAlert: false) {
                            let url = try manager.requestSSOURL()
                            if !NSWorkspace.shared.open(url) {
                                throw URLError(.badURL)
                            }
                        }
                    }
                    .onReceive(AppOpenURLManager.shared.publisher.receive(on: DispatchQueue.main)) { urlString in
                        do {
                            let token = try LicenseManager.extractSSOToken(from: urlString, scheme: "licensespring")
                            showAlert(
                                title: "SSO Activation", text: "SSO activation token received",
                                style: .informational
                            )
                            
                            _ = try manager.activateLicense(ssoToken: token)
                            viewUpdater.update()
                        } catch {
                            showAlert(title: "SSO Activation", text: error.localizedDescription, style: .critical)
                        }
                    }
                }
                .frame(maxWidth: .infinity)
            }
            GroupBox(label: Text("Air-gap Activation")) {
                HStack(alignment: .top) {
                    VStack {
                        TextField("License Key", text: $airgapLicenseKey)
                        TextField("Initialization Code", text: $initializationCode)
                        HStack {
                            MinWidthButton("Generate Activation Code", width: 168) {
                                performAction(name: "Air-gap Activation", successAlert: false) {
                                    activationCode = try manager.generateAirGapActivationCode(
                                        initializationCode: initializationCode,
                                        licenseKey: airgapLicenseKey
                                    )
                                }
                            }
                            .disabled(airgapLicenseKey.isEmpty || initializationCode.isEmpty)
                            MinWidthButton("Open Activation Page", width: 168) {
                                NSWorkspace.shared.open(URL(string: "https://airgap.licensespring.com")!)
                            }
                        }
                        Text("Activation Code")
                            .frame(maxWidth: .infinity, alignment: .leading)
                        TextField("Activation Code", text: .constant(activationCode))
                        Text("Hardware ID")
                            .frame(maxWidth: .infinity, alignment: .leading)
                        TextField("Hardware ID", text: .constant(activationCode.isEmpty ? "" : manager.configuration.hardwareID))
                    }
                    VStack {
                        Text("Policy File")
                            .frame(maxWidth: .infinity, alignment: .leading)
                        HStack {
                            TextField("Policy File", text: .constant(policyFile?.path ?? ""))
                            Button("Select...") { policyFile = locationForFile(save: false) }
                        }
                        Text("Policy ID")
                            .frame(maxWidth: .infinity, alignment: .leading)
                        TextField("Policy ID", text: $policyID)
                        Text("Confirmation Code")
                            .frame(maxWidth: .infinity, alignment: .leading)
                        TextField("Confirmation Code", text: $confirmationCode)
                        
                        MinWidthButton("Activate") {
                            performAction(name: "Air-gap Activation") {
                                guard let policyFile, let policyID = Int(policyID) else { return }
                                try manager.verifyAirGapConfirmationCode(
                                    confirmationCode: confirmationCode,
                                    licenseKey: airgapLicenseKey,
                                    policyID: policyID
                                )
                                _ = try manager.activateLicense(
                                    airgapPolicyFile: policyFile,
                                    licenseKey: airgapLicenseKey,
                                    policyID: policyID
                                )
                            }
                        }
                        .disabled(Int(policyID) == nil || policyFile == nil || confirmationCode.isEmpty)
                    }
                }
                .frame(maxWidth: .infinity)
            }
            GroupBox(label: Text("Bundle Activation")){
                if let bundleManager = bundleManager, bundleManager.configuration.isVMDetectionEnabled {
                    Text("Virtual machine \(bundleManager.configuration.isVM ? "detected" : "not detected")")
                        .foregroundColor(.red)
                        .fontWeight(.bold)
                        .frame(alignment: .leading)
                }
                HStack(alignment: .top) {
                    VStack {
                        Text("Key-based product")
                        TextField("License Key", text: $bundleLicenseKey)
                        TextField("Invisible", text: .constant(""))
                            .opacity(0)
                        MinWidthButton("Activate") {
                            performAction(name: "Activate License", successAlert: false) {
                                _ = try bundleManager.activateBundle(licenseKey: bundleLicenseKey)
                            }
                        }
                        .disabled(bundleLicenseKey.isEmpty)
                        MinWidthButton("Create Request") {
                            guard let location = locationForFile(save: true) else { return }
                            performAction(name: "Create Offline Activation") {
                                _ = try bundleManager.createOfflineActivationFile(licenseKey: bundleLicenseKey, preferredLocation: location)
                            }
                        }
                        .disabled(bundleLicenseKey.isEmpty)
                    }
                    Divider()
                    VStack {
                        Text("User-based product")
                        TextField("User", text: $bundleUser)
                        TextField("Password", text: $bundlePassword)
                        TextField("Server ID", text: $bundleServerId)
                        MinWidthButton("Activate") {
                            performAction(name: "Activate License", successAlert: false) {
                                if let serverId = Int64(bundleServerId){
                                    _ = try bundleManager.activateBundle(username: bundleUser, password: bundlePassword, serverId: serverId)
                                }
                                else {
                                    _ = try bundleManager.activateBundle(username: bundleUser, password: bundlePassword)
                                }
                            }
                        }
                        .disabled(bundleUser.isEmpty || bundlePassword.isEmpty)
                        MinWidthButton("Create Request") {
                            guard let location = locationForFile(save: true) else { return }
                            performAction(name: "Create Offline Activation") {
                                _ = try bundleManager.createOfflineActivationFile(user: bundleUser, password: bundlePassword, preferredLocation: location, serverID: Int64(bundleServerId) ?? 0)
                            }
                        }
                        .disabled(user.isEmpty || password.isEmpty)
                    }
                    //what about idToken and code
                    //activateBundle(idToken: String)
                    //activateBundle(code: String)
                }
            }
            GroupBox(label: Text("Offline Activation")) {
                HStack(alignment: .top) {
                    MinWidthButton("Open Activation Page") {
                        NSWorkspace.shared.open(URL(string: "https://saas.licensespring.com/offline/")!)
                    }
                    MinWidthButton("Activate Offline") {
                        guard let location = locationForFile(save: false) else { return }
                        performAction(name: "Activate Offline") {
                            _ = try bundleManager.activateBundleOffline(location)
                        }
                    }
                }
                .frame(maxWidth: .infinity)
            }
        }
    }
}



struct InitialView_Previews: PreviewProvider {
    static var previews: some View {
        InitialView(manager: nil, bundleManager: nil, showConfiguration: .constant(false))
            .frame(height: 1000)
    }
}

